From Ltac2 Require Import Ltac2.
From Hammer Require Import Tactics.

Class Lattice (A : Set) := {
    meet : A -> A -> A;
    join : A -> A -> A;
    meet_commutative : forall a b, meet a b = meet b a;
    meet_associative : forall a b c, meet (meet a b) c = meet a (meet b c);
    meet_absorptive : forall a b, meet a (join a b) = a;
    meet_idempotent : forall a, meet a a = a;
    join_commutative : forall a b, join a b = join b a;
    join_associative: forall a b c, join (join a b) c = join a (join b c);
    join_absorptive : forall a b, join a (meet a b) = a;
    join_idempotent : forall a, join a a = a;
  }.

Generalizable Variable A.

From Coq Require Import Classes.RelationClasses.

Definition leq_lat `{Lattice A} (a : A) (b : A) :=
  meet a b = a.

Add Parametric Relation `(EL : Lattice A) : A leq_lat
    reflexivity proved by ltac2:(ltac1:(sauto))
    transitivity proved by ltac2:(ltac1:(scongruence
                                           use: @meet_commutative,
                                            @meet_associative,
                                            @meet_idempotent
                                              unfold: leq_lat))
    as leq_lat_rewr.

Lemma leq_lat_antisym `{Lattice A} (e1 e2 : A) :
  leq_lat e1 e2 -> leq_lat e2 e1 -> e1 = e2.
Proof.
  ltac1:(scongruence use: @meet_commutative, @meet_idempotent unfold: leq_lat).
Qed.


(* sqcap *)
Infix "⊓" := meet (at level 40, left associativity).
(* sqcup *)
Infix "⊔" := join (at level 36, left associativity).

Infix "≤" := leq_lat (at level 50).

Inductive relevance :=
| q_R : relevance
| q_I : relevance.

Inductive fragment :=
| t_L : fragment
| t_P : fragment.

(* If an equality typechecks under D, then it is treated as a plain type *)
Inductive diagonality :=
| t_ND : diagonality
| t_D : diagonality.

Set Default Proof Mode "Classic".
#[refine, export] Instance rel_Lattice : Lattice relevance := {
    meet := fun x y =>
              match x with
              | q_R => q_R
              | _ => y
              end ;
    join := fun x y =>
              match x with
              | q_I => q_I
              | _ => y
              end;
  }.
Proof.
  all:try solve [qauto inv:relevance].
Defined.

#[refine, export] Instance frag_Lattice : Lattice fragment := {
    meet := fun x y =>
              match x with
              | t_L => t_L
              | _ => y
              end ;
    join := fun x y =>
              match x with
              | t_P => t_P
              | _ => y
              end;
  }.
Proof.
  all:try solve [qauto inv:fragment].
Defined.

#[refine, export] Instance diag_Lattice : Lattice diagonality := {
    meet := fun x y =>
              match x with
              | t_ND => t_ND
              | _ => y
              end ;
    join := fun x y =>
              match x with
              | t_D => t_D
              | _ => y
              end;
  }.
Proof.
  all:try solve [qauto inv:fragment].
Defined.

(* product of lattices with point-wise comparison  *)
#[refine, export]Instance prod_lattice {A B} `{Lattice A} `{Lattice B} : Lattice (A * B) := {
    meet := fun x y =>
              match x, y with
              | (x0, x1), (y0, y1) =>
                  ((x0 ⊓ y0), (x1 ⊓ y1))
              end;
    join := fun x y =>
              match x, y with
              | (x0, x1), (y0, y1) =>
                  ((x0 ⊔ y0), (x1 ⊔ y1))
              end
  }.
Proof.
  all:try solve [hauto lq: on rew: off inv: prod, Lattice].
Defined.

Lemma prod_lattice_leq_iff {A B : Set} `{Lattice A} `{Lattice B} (a1 a2 : A) (b1 b2 : B) :
  (a1 , b1) ≤ (a2 , b2) <-> a1 ≤ a2 /\ b1 ≤ b2.
Proof.
  sfirstorder unfold:leq_lat.
Qed.

(* We need these trivial lemmas so we can still reason about these
functions after setting them as opaque *)

(* Lemmas that characterize the relevance lattice *)
Lemma not_q_I_leq_q_R : not (q_I ≤ q_R).
Proof.
  sauto q:on.
Qed.

Lemma rel_meet_R1 (x : relevance) : x ⊓ q_R = q_R.
Proof.
  hauto lq:on.
Qed.

Lemma rel_meet_R2 (x : relevance) : q_R ⊓ x = q_R.
Proof.
  hauto lq:on.
Qed.

Lemma rel_join_I1 (x : relevance) : x ⊔ q_I = q_I.
Proof.
  hauto lq:on.
Qed.

Lemma rel_join_I2 (x : relevance) : q_I ⊔ x = q_I.
Proof.
  hauto lq:on.
Qed.

#[export] Hint Rewrite -> rel_meet_R1 rel_meet_R2 rel_join_I1 rel_join_I2 : core.

Lemma rel_leq_I (x : relevance) : x ≤ q_I.
Proof.
  sauto lq:on.
Qed.

Lemma rel_leq_R (x : relevance) : q_R ≤ x.
Proof.
  sauto lq:on.
Qed.

#[export] Hint Resolve rel_leq_I rel_leq_R not_q_I_leq_q_R : core.

Require Import Coq.Logic.Decidable.
Lemma relevance_leq_dec : forall (a b : relevance), decidable (leq_lat a b).
Proof.
  sauto q:on.
Qed.

(* Lemmas that characterize the fragment lattice *)
Lemma frag_meet_L1 (x : fragment) : x ⊓ t_L = t_L.
Proof.
  hauto lq:on.
Qed.

Lemma frag_meet_L2 (x : fragment) : t_L ⊓ x = t_L.
Proof.
  hauto lq:on.
Qed.

Lemma frag_join_P1 (x : fragment) : x ⊔ t_P = t_P.
Proof.
  hauto lq:on.
Qed.

Lemma frag_join_P2 (x : fragment) : t_P ⊔ x = t_P.
Proof.
  hauto lq:on.
Qed.

Lemma frag_leq_P (x : fragment) : x ≤ t_P.
Proof.
  sauto lq:on.
Qed.

Lemma frag_leq_L (x : fragment) : t_L ≤ x.
Proof.
  sauto lq:on.
Qed.

(* Lemmas that characterize the diagonality lattice *)
Lemma diag_meet_ND1 (x : diagonality) : x ⊓ t_ND = t_ND.
Proof.
  hauto lq:on.
Qed.

Lemma diag_meet_ND2 (x : diagonality) : t_ND ⊓ x = t_ND.
Proof.
  hauto lq:on.
Qed.

Lemma diag_join_D1 (x : diagonality) : x ⊔ t_D = t_D.
Proof.
  hauto lq:on.
Qed.

Lemma diag_join_D2 (x : diagonality) : t_D ⊔ x = t_D.
Proof.
  hauto lq:on.
Qed.

Lemma diag_leq_D (x : diagonality) : x ≤ t_D.
Proof.
  sauto lq:on.
Qed.

Lemma diag_leq_ND (x : diagonality) : t_ND ≤ x.
Proof.
  sauto lq:on.
Qed.

(* Lemmas that characterize the product construction *)
Lemma prod_lattice_eq_meet {A B} `{Lattice A} `{Lattice B}
  (x0 x1: A) (y0 y1 : B) : (x0,y0) ⊓ (x1,y1) = (x0 ⊓ x1, y0 ⊓ y1).
Proof.
  trivial.
Qed.

Lemma prod_lattice_eq_join {A B} `{Lattice A} `{Lattice B}
  (x0 x1: A) (y0 y1 : B) : (x0,y0) ⊔ (x1,y1) = (x0 ⊔ x1, y0 ⊔ y1).
Proof.
  trivial.
Qed.

Lemma prod_lattice_leq {A B} `{Lattice A} `{Lattice B}
  (x0 x1: A) (y0 y1 : B) : (x0,y0) ≤ (x1,y1) <-> x0 ≤ x1 /\ y0 ≤ y1.
Proof.
  sfirstorder use: @prod_lattice_eq_meet unfold: leq_lat.
Qed.
